<?php

namespace App\Controllers;

use App\Models\UserModel;
use CodeIgniter\Config\Factories;

class InstallController extends BaseController
{
    private function lockPath(): string
    {
        return WRITEPATH . 'install/installed.lock';
    }

    public function index()
    {
        if (is_file($this->lockPath())) {
            return $this->render('install/already');
        }

        if ($this->request->getMethod() === 'post') {
            $rules = [
                'db_host'        => 'required',
                'db_name'        => 'required',
                'db_user'        => 'required',
                'db_port'        => 'required|integer',
                'admin_name'     => 'required|min_length[2]',
                'admin_email'    => 'required|valid_email',
                'admin_password' => 'required|min_length[6]',
                'base_url'       => 'permit_empty|valid_url_strict',
            ];

            if (! $this->validate($rules)) {
                return $this->render('install/index', ['validation' => $this->validator]);
            }

            $dbHost = (string) $this->request->getPost('db_host');
            $dbName = (string) $this->request->getPost('db_name');
            $dbUser = (string) $this->request->getPost('db_user');
            $dbPass = (string) $this->request->getPost('db_pass');
            $dbPort = (int) $this->request->getPost('db_port');
            $baseUrl = trim((string) $this->request->getPost('base_url'));

            $mysqli = @new \mysqli($dbHost, $dbUser, $dbPass, '', $dbPort);
            if ($mysqli->connect_errno) {
                return redirect()->back()->withInput()->with('error', 'DB connection failed: ' . $mysqli->connect_error);
            }

            if (! $mysqli->query("CREATE DATABASE IF NOT EXISTS `{$dbName}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci")) {
                return redirect()->back()->withInput()->with('error', 'Could not create database: ' . $mysqli->error);
            }

            $cfg = [
                'hostname' => $dbHost,
                'database' => $dbName,
                'username' => $dbUser,
                'password' => $dbPass,
                'DBDriver' => 'MySQLi',
                'DBPrefix' => '',
                'port'     => $dbPort,
                'charset'  => 'utf8mb4',
                'DBCollat' => 'utf8mb4_unicode_ci',
            ];

            if (! is_dir(WRITEPATH . 'install')) {
                mkdir(WRITEPATH . 'install', 0775, true);
            }

            file_put_contents(WRITEPATH . 'install/db_config.php', "<?php\nreturn " . var_export($cfg, true) . ";\n");

            try {
                $db = \Config\Database::connect('default', false);
                $migrations = service('migrations');
                $migrations->setSilent(false);
                $migrations->latest();

                $seeder = service('seeder');
                $seeder->call('ServiceTypeSeeder');

                $users = new UserModel();
                $exists = $users->where('email', (string) $this->request->getPost('admin_email'))->first();
                if (! $exists) {
                    $users->insert([
                        'name'          => (string) $this->request->getPost('admin_name'),
                        'email'         => (string) $this->request->getPost('admin_email'),
                        'password_hash' => password_hash((string) $this->request->getPost('admin_password'), PASSWORD_DEFAULT),
                        'role'          => 'admin',
                        'is_active'     => 1,
                    ]);
                }
            } catch (\Throwable $e) {
                return redirect()->back()->withInput()->with('error', 'Installation failed: ' . $e->getMessage());
            }

            if ($baseUrl !== '' && is_file(ROOTPATH . '.env')) {
                $env = file_get_contents(ROOTPATH . '.env');
                if ($env !== false) {
                    $safeUrl = rtrim($baseUrl, '/') . '/';
                    $env = preg_replace("/^app\\.baseURL\\s*=.*$/m", "app.baseURL = '" . $safeUrl . "'", $env);
                    file_put_contents(ROOTPATH . '.env', $env);
                }
            }

            file_put_contents($this->lockPath(), 'installed_at=' . date('c'));

            return redirect()->to('/login')->with('success', 'Installation completed.');
        }

        return $this->render('install/index');
    }
}
