<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\ProjectModel;
use App\Models\ServiceTypeModel;
use App\Models\TaskModel;
use App\Models\UserModel;

class TasksController extends BaseController
{
    public function index()
    {
        $status = $this->request->getGet('status');
        $assignee = $this->request->getGet('assignee');
        $project = $this->request->getGet('project');

        $builder = (new TaskModel())
            ->select('tasks.*, projects.name AS project_name, users.name AS assignee_name, service_types.name AS service_type_name')
            ->join('projects', 'projects.id = tasks.project_id')
            ->join('users', 'users.id = tasks.assigned_to', 'left')
            ->join('service_types', 'service_types.id = tasks.service_type_id');

        if ($status) {
            $builder->where('tasks.status', $status);
        }
        if ($assignee) {
            $builder->where('tasks.assigned_to', $assignee);
        }
        if ($project) {
            $builder->where('tasks.project_id', $project);
        }

        $tasks = $builder->orderBy('tasks.id', 'desc')->findAll();

        return $this->render('admin/tasks/index', [
            'tasks'    => $tasks,
            'projects' => (new ProjectModel())->findAll(),
            'staff'    => (new UserModel())->where('role', 'staff')->findAll(),
        ]);
    }

    public function create()
    {
        if ($this->request->getMethod() === 'post') {
            $rules = [
                'project_id'      => 'required|integer',
                'title'           => 'required|min_length[2]',
                'service_type_id' => 'required|integer',
                'priority'        => 'required|in_list[low,medium,high]',
                'status'          => 'required|in_list[todo,doing,done]',
            ];

            if (! $this->validate($rules)) {
                return $this->render('admin/tasks/form', $this->formData(null, $this->validator));
            }

            (new TaskModel())->insert([
                'project_id'      => $this->request->getPost('project_id'),
                'title'           => $this->request->getPost('title'),
                'description'     => $this->request->getPost('description'),
                'service_type_id' => $this->request->getPost('service_type_id'),
                'priority'        => $this->request->getPost('priority'),
                'status'          => $this->request->getPost('status'),
                'due_date'        => $this->request->getPost('due_date') ?: null,
                'assigned_to'     => $this->request->getPost('assigned_to') ?: null,
                'created_by'      => session()->get('user_id'),
            ]);

            return redirect()->to('/admin/tasks')->with('success', 'Task created.');
        }

        return $this->render('admin/tasks/form', $this->formData(null));
    }

    public function edit(int $id)
    {
        $model = new TaskModel();
        $task = $model->find($id);

        if (! $task) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound();
        }

        if ($this->request->getMethod() === 'post') {
            $rules = [
                'project_id'      => 'required|integer',
                'title'           => 'required|min_length[2]',
                'service_type_id' => 'required|integer',
                'priority'        => 'required|in_list[low,medium,high]',
                'status'          => 'required|in_list[todo,doing,done]',
            ];

            if (! $this->validate($rules)) {
                return $this->render('admin/tasks/form', $this->formData($task, $this->validator));
            }

            $model->update($id, [
                'project_id'      => $this->request->getPost('project_id'),
                'title'           => $this->request->getPost('title'),
                'description'     => $this->request->getPost('description'),
                'service_type_id' => $this->request->getPost('service_type_id'),
                'priority'        => $this->request->getPost('priority'),
                'status'          => $this->request->getPost('status'),
                'due_date'        => $this->request->getPost('due_date') ?: null,
                'assigned_to'     => $this->request->getPost('assigned_to') ?: null,
            ]);

            return redirect()->to('/admin/tasks')->with('success', 'Task updated.');
        }

        return $this->render('admin/tasks/form', $this->formData($task));
    }

    private function formData(?array $task, $validation = null): array
    {
        return [
            'task'         => $task,
            'validation'   => $validation,
            'projects'     => (new ProjectModel())->findAll(),
            'serviceTypes' => (new ServiceTypeModel())->where('is_active', 1)->findAll(),
            'staff'        => (new UserModel())->where('role', 'staff')->where('is_active', 1)->findAll(),
        ];
    }
}
