<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\CustomerModel;
use App\Models\ProjectModel;

class ProjectsController extends BaseController
{
    public function index()
    {
        $projects = (new ProjectModel())
            ->select('projects.*, customers.company_name')
            ->join('customers', 'customers.id = projects.customer_id')
            ->orderBy('projects.id', 'desc')
            ->findAll();

        return $this->render('admin/projects/index', ['projects' => $projects]);
    }

    public function create()
    {
        $projectModel = new ProjectModel();
        $customerModel = new CustomerModel();

        if ($this->request->getMethod() === 'post') {
            $rules = [
                'customer_id' => 'required|integer',
                'name'        => 'required|min_length[2]',
                'status'      => 'required|in_list[planned,active,done,cancelled]',
            ];

            if (! $this->validate($rules)) {
                return $this->render('admin/projects/form', [
                    'project'    => null,
                    'customers'  => $customerModel->findAll(),
                    'validation' => $this->validator,
                ]);
            }

            $projectModel->insert([
                'customer_id' => $this->request->getPost('customer_id'),
                'name'        => $this->request->getPost('name'),
                'description' => $this->request->getPost('description'),
                'status'      => $this->request->getPost('status'),
                'start_date'  => $this->request->getPost('start_date') ?: null,
                'due_date'    => $this->request->getPost('due_date') ?: null,
                'created_by'  => session()->get('user_id'),
            ]);

            return redirect()->to('/admin/projects')->with('success', 'Project created.');
        }

        return $this->render('admin/projects/form', [
            'project'   => null,
            'customers' => $customerModel->findAll(),
        ]);
    }

    public function edit(int $id)
    {
        $projectModel = new ProjectModel();
        $customerModel = new CustomerModel();
        $project = $projectModel->find($id);

        if (! $project) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound();
        }

        if ($this->request->getMethod() === 'post') {
            $rules = [
                'customer_id' => 'required|integer',
                'name'        => 'required|min_length[2]',
                'status'      => 'required|in_list[planned,active,done,cancelled]',
            ];

            if (! $this->validate($rules)) {
                return $this->render('admin/projects/form', [
                    'project'    => $project,
                    'customers'  => $customerModel->findAll(),
                    'validation' => $this->validator,
                ]);
            }

            $projectModel->update($id, [
                'customer_id' => $this->request->getPost('customer_id'),
                'name'        => $this->request->getPost('name'),
                'description' => $this->request->getPost('description'),
                'status'      => $this->request->getPost('status'),
                'start_date'  => $this->request->getPost('start_date') ?: null,
                'due_date'    => $this->request->getPost('due_date') ?: null,
            ]);

            return redirect()->to('/admin/projects')->with('success', 'Project updated.');
        }

        return $this->render('admin/projects/form', [
            'project'   => $project,
            'customers' => $customerModel->findAll(),
        ]);
    }

    public function delete(int $id)
    {
        (new ProjectModel())->delete($id);
        return redirect()->to('/admin/projects')->with('success', 'Project deleted.');
    }
}
