<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\CustomerInvoiceModel;
use App\Models\CustomerModel;
use App\Models\CustomerPaymentModel;
use App\Models\ProjectModel;

class CustomersController extends BaseController
{
    public function index()
    {
        $customers = (new CustomerModel())->orderBy('id', 'desc')->findAll();
        return $this->render('admin/customers/index', ['customers' => $customers]);
    }

    public function create()
    {
        $model = new CustomerModel();

        if ($this->request->getMethod() === 'post') {
            $rules = [
                'company_name' => 'required|min_length[2]',
                'email'        => 'permit_empty|valid_email',
            ];

            if (! $this->validate($rules)) {
                return $this->render('admin/customers/form', ['validation' => $this->validator, 'customer' => null]);
            }

            $model->insert([
                'company_name' => $this->request->getPost('company_name'),
                'contact_name' => $this->request->getPost('contact_name'),
                'phone'        => $this->request->getPost('phone'),
                'email'        => $this->request->getPost('email'),
                'tax_office'   => $this->request->getPost('tax_office'),
                'tax_number'   => $this->request->getPost('tax_number'),
                'address'      => $this->request->getPost('address'),
                'notes'        => $this->request->getPost('notes'),
                'is_active'    => (int) $this->request->getPost('is_active') === 0 ? 0 : 1,
            ]);

            return redirect()->to('/admin/customers')->with('success', 'Customer created.');
        }

        return $this->render('admin/customers/form', ['customer' => null]);
    }

    public function edit(int $id)
    {
        $model = new CustomerModel();
        $customer = $model->find($id);

        if (! $customer) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound();
        }

        if ($this->request->getMethod() === 'post') {
            $rules = [
                'company_name' => 'required|min_length[2]',
                'email'        => 'permit_empty|valid_email',
            ];

            if (! $this->validate($rules)) {
                return $this->render('admin/customers/form', ['validation' => $this->validator, 'customer' => $customer]);
            }

            $model->update($id, [
                'company_name' => $this->request->getPost('company_name'),
                'contact_name' => $this->request->getPost('contact_name'),
                'phone'        => $this->request->getPost('phone'),
                'email'        => $this->request->getPost('email'),
                'tax_office'   => $this->request->getPost('tax_office'),
                'tax_number'   => $this->request->getPost('tax_number'),
                'address'      => $this->request->getPost('address'),
                'notes'        => $this->request->getPost('notes'),
                'is_active'    => (int) $this->request->getPost('is_active') === 0 ? 0 : 1,
            ]);

            return redirect()->to('/admin/customers')->with('success', 'Customer updated.');
        }

        return $this->render('admin/customers/form', ['customer' => $customer]);
    }

    public function show(int $id)
    {
        $customerModel = new CustomerModel();
        $projectModel = new ProjectModel();
        $invoiceModel = new CustomerInvoiceModel();
        $paymentModel = new CustomerPaymentModel();

        $customer = $customerModel->find($id);
        if (! $customer) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound();
        }

        $projects = $projectModel->where('customer_id', $id)->orderBy('id', 'desc')->findAll();
        $invoices = $invoiceModel->where('customer_id', $id)->orderBy('date', 'desc')->findAll();
        $payments = $paymentModel->where('customer_id', $id)->orderBy('date', 'desc')->findAll();

        $invoiceTotal = (float) ($invoiceModel->selectSum('amount')->where('customer_id', $id)->first()['amount'] ?? 0);
        $paymentTotal = (float) ($paymentModel->selectSum('amount')->where('customer_id', $id)->first()['amount'] ?? 0);

        $timeline = [];
        foreach ($invoices as $inv) {
            $timeline[] = ['type' => 'invoice', 'date' => $inv['date'], 'title' => $inv['title'], 'amount' => $inv['amount']];
        }
        foreach ($payments as $pay) {
            $timeline[] = ['type' => 'payment', 'date' => $pay['date'], 'title' => strtoupper((string) ($pay['method'] ?? 'payment')), 'amount' => $pay['amount']];
        }
        usort($timeline, static fn($a, $b) => strcmp($b['date'], $a['date']));

        return $this->render('admin/customers/show', [
            'customer'     => $customer,
            'projects'     => $projects,
            'invoices'     => $invoices,
            'payments'     => $payments,
            'timeline'     => $timeline,
            'invoiceTotal' => $invoiceTotal,
            'paymentTotal' => $paymentTotal,
            'balance'      => $invoiceTotal - $paymentTotal,
        ]);
    }

    public function createInvoice(int $customerId)
    {
        $rules = [
            'date'   => 'required|valid_date',
            'title'  => 'required',
            'amount' => 'required|decimal',
        ];

        if (! $this->validate($rules)) {
            return redirect()->back()->with('error', 'Invoice validation failed.');
        }

        (new CustomerInvoiceModel())->insert([
            'customer_id' => $customerId,
            'project_id'  => $this->request->getPost('project_id') ?: null,
            'date'        => $this->request->getPost('date'),
            'title'       => $this->request->getPost('title'),
            'amount'      => $this->request->getPost('amount'),
            'vat_rate'    => $this->request->getPost('vat_rate') ?: null,
            'vat_amount'  => $this->request->getPost('vat_amount') ?: null,
            'note'        => $this->request->getPost('note'),
            'created_by'  => session()->get('user_id'),
        ]);

        return redirect()->to('/admin/customers/' . $customerId)->with('success', 'Invoice added.');
    }

    public function createPayment(int $customerId)
    {
        $rules = [
            'date'   => 'required|valid_date',
            'amount' => 'required|decimal',
        ];

        if (! $this->validate($rules)) {
            return redirect()->back()->with('error', 'Payment validation failed.');
        }

        (new CustomerPaymentModel())->insert([
            'customer_id' => $customerId,
            'project_id'  => $this->request->getPost('project_id') ?: null,
            'date'        => $this->request->getPost('date'),
            'amount'      => $this->request->getPost('amount'),
            'vat_rate'    => $this->request->getPost('vat_rate') ?: null,
            'vat_amount'  => $this->request->getPost('vat_amount') ?: null,
            'method'      => $this->request->getPost('method') ?: null,
            'note'        => $this->request->getPost('note'),
            'created_by'  => session()->get('user_id'),
        ]);

        return redirect()->to('/admin/customers/' . $customerId)->with('success', 'Payment added.');
    }
}
